<?php
declare(strict_types=1);

namespace App\Controller;

/**
 * Products Controller
 *
 * @property \App\Model\Table\ProductsTable $Products
 */
class ProductsController extends AppController
{
    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     */
    public function beforeFilter(\Cake\Event\EventInterface $event)
    {
        parent::beforeFilter($event);
        $this->Authentication->allowUnauthenticated(['detail']);
    }
    private function isAdmin(): bool
    {
        $id = $this->request->getAttribute('identity');
        return (bool)($id && $id->get('role') === 'admin');
    }
    public function index()
    {
        if (!$this->isAdmin()) {
            throw new \Cake\Http\Exception\ForbiddenException('Admins only.');
        }
        $query = $this->Products->find()->where(['archived' => 'No']);
        $products = $this->paginate($query);
        $this->set(compact('products'));
    }

    public function archieved()
    {
        if (!$this->isAdmin()) {
            throw new \Cake\Http\Exception\ForbiddenException('Admins only.');
        }
        $query = $this->Products->find()->where(['archived' => 'Yes']);
        $products = $this->paginate($query);

        $this->set(compact('products'));
    }

    public function archive($id = null)
    {
        if (!$this->isAdmin()) {
            throw new \Cake\Http\Exception\ForbiddenException('Admins only.');
        }
        $this->request->allowMethod(['post']);

        $this->Products->updateAll(
            ['archived' => 'Yes'],
            ['id' => $id]
        );
        $this->Flash->success(__('The product has been archived.'));
        return $this->redirect(['action' => 'index']);
    }

    public function unArchive($id = null)
    {
        if (!$this->isAdmin()) {
            throw new \Cake\Http\Exception\ForbiddenException('Admins only.');
        }
        $this->request->allowMethod(['post']);

        $this->Products->updateAll(
            ['archived' => 'No'],
            ['id' => $id]
        );
        $this->Flash->success(__('The product has been unarchived.'));
        return $this->redirect(['action' => 'index']);
    }
    public function detail($id=null){
        $product = $this->Products->get($id);
        $this->set(compact('product'));
    }

    /**
     * View method
     *
     * @param string|null $id Product id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)
    {
        if (!$this->isAdmin()) {
            throw new \Cake\Http\Exception\ForbiddenException('Admins only.');
        }
        $product = $this->Products->get($id);
        $this->set(compact('product'));
    }
    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function add()
    {
        if (!$this->isAdmin()) {
            throw new \Cake\Http\Exception\ForbiddenException('Admins only.');
        }
        $product = $this->Products->newEmptyEntity();
        if ($this->request->is('post')) {
            $data = $this->request->getData();
            $imageUrls = [];
            $productImgDir = WWW_ROOT . 'img' . DS . 'products';
            if (!is_dir($productImgDir)) {
                mkdir($productImgDir, 0777, true);
            }
            if (!empty($_FILES['image_url'])) {
                if (is_array($_FILES['image_url']['name'])) {
                    $fileCount = count($_FILES['image_url']['name']);
                    for ($i = 0; $i < $fileCount; $i++) {
                        if ($_FILES['image_url']['error'][$i] === UPLOAD_ERR_OK) {
                            $originalName = $_FILES['image_url']['name'][$i];
                            $extension = pathinfo($originalName, PATHINFO_EXTENSION);
                            $uniqueName = time() . '_' . uniqid() . '_' . $i . '.' . $extension;
                            $destination = $productImgDir . DS . $uniqueName;
                            if (move_uploaded_file($_FILES['image_url']['tmp_name'][$i], $destination)) {
                                $imageUrls[] = '/team212-app_fit3047/img/products/' . $uniqueName;
                            }
                        }
                    }
                } else {
                    if ($_FILES['image_url']['error'] === UPLOAD_ERR_OK) {
                        $originalName = $_FILES['image_url']['name'];
                        $extension = pathinfo($originalName, PATHINFO_EXTENSION);
                        $uniqueName = time() . '_' . uniqid() . '.' . $extension;
                        $destination = $productImgDir . DS . $uniqueName;
                        if (move_uploaded_file($_FILES['image_url']['tmp_name'], $destination)) {
                            $imageUrls[] = '/team212-app_fit3047/img/products/' . $uniqueName;
                        }
                    }
                }
            }
            $data['image_url'] = json_encode($imageUrls, JSON_UNESCAPED_SLASHES);
            $product = $this->Products->patchEntity($product, $data, [
                'fieldList' => ['name','price','sale_price','description','category','image_url','sku','stock','threshold','archived']
            ]);
            if ($this->Products->save($product)) {
                $this->Flash->success(__('The product has been saved.'));
                return $this->redirect(['action' => 'index']);
            }
            debug($product->getErrors());
            $this->Flash->error(__('The product could not be saved. Please, try again.'));
        }
        $this->set(compact('product'));
    }



    /**
     * Edit method
     *
     * @param string|null $id Product id.
     * @return \Cake\Http\Response|null|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function edit($id = null)
    {
        if (!$this->isAdmin()) {
            throw new \Cake\Http\Exception\ForbiddenException('Admins only.');
        }
        $product = $this->Products->get($id, contain: []);
        if ($this->request->is(['patch', 'post', 'put'])) {
            $data = $this->request->getData();
            $imageUrls = [];
            if (!empty($_FILES['image_url'])) {
                $productImgDir = WWW_ROOT . 'img' . DS . 'products';
                if (!is_dir($productImgDir)) {
                    mkdir($productImgDir, 0777, true);
                }
                if (is_array($_FILES['image_url']['name'])) {
                    $fileCount = count($_FILES['image_url']['name']);
                    for ($i = 0; $i < $fileCount; $i++) {
                        if ($_FILES['image_url']['error'][$i] === UPLOAD_ERR_OK) {
                            $originalName = $_FILES['image_url']['name'][$i];
                            $extension = pathinfo($originalName, PATHINFO_EXTENSION);
                            $uniqueName = time() . '_' . uniqid() . '_' . $i . '.' . $extension;
                            $destination = $productImgDir . DS . $uniqueName;
                            if (move_uploaded_file($_FILES['image_url']['tmp_name'][$i], $destination)) {
                                $relativePath = '/team212-app_fit3047/img/products/' . $uniqueName;
                                $imageUrls[] = $relativePath;
                            }
                        }
                    }
                } else {
                    if ($_FILES['image_url']['error'] === UPLOAD_ERR_OK) {
                        $originalName = $_FILES['image_url']['name'];
                        $extension = pathinfo($originalName, PATHINFO_EXTENSION);
                        $uniqueName = time() . '_' . uniqid() . '.' . $extension;
                        $destination = $productImgDir . DS . $uniqueName;
                        if (move_uploaded_file($_FILES['image_url']['tmp_name'], $destination)) {
                            $relativePath = '/team212-app_fit3047/img/products/' . $uniqueName;
                            $imageUrls[] = $relativePath;
                        }
                    }
                }
                $deleteImages = !empty($data['delete_images']) ? (array)$data['delete_images'] : [];
                $currentImages = $product->image_url ?: [];
                $currentImages = array_diff($currentImages, $deleteImages);
                $data['image_url'] = array_merge($currentImages, $imageUrls);
            } else {
                $deleteImages = !empty($data['delete_images']) ? (array)$data['delete_images'] : [];
                $data['image_url'] = array_diff($product->image_url ?: [], $deleteImages);
            }
            $product = $this->Products->patchEntity($product, $data);
            if ($this->Products->save($product)) {
                $this->Flash->success(__('The product has been saved.'));
                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The product could not be saved. Please, try again.'));
        }
        $this->set(compact('product'));
    }

    /**
     * Delete method
     *
     * @param string|null $id Product id.
     * @return \Cake\Http\Response|null Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function delete($id = null)
    {
        if (!$this->isAdmin()) {
            throw new \Cake\Http\Exception\ForbiddenException('Admins only.');
        }
        $this->request->allowMethod(['post', 'delete']);
        $product = $this->Products->get($id);
        if ($this->Products->delete($product)) {
            $this->Flash->success(__('The product has been deleted.'));
        } else {
            $this->Flash->error(__('The product could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'archieved']);
    }

    public function addToCart($id = null)
    {
        $this->request->allowMethod(['post', 'get']);

        $product = $this->Products->findById($id)->first();
        if (!$product) {
            throw new NotFoundException('Product not found');
        }
        $session = $this->request->getSession();
        $cart = (array)$session->read('Cart');
        $cart[$product->id] = ($cart[$product->id] ?? 0) + 1;
        $session->write('Cart', $cart);

        $this->Flash->success(__('{0} added to cart.', $product->name));
        $product->stock = $product->stock - $cart[$product->id];
        return $this->redirect($this->referer(['action' => 'index'], true));
    }
}
