<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;

class OrdersTable extends Table
{
    public function initialize(array $config): void
    {
        parent::initialize($config);

        // DB table
        $this->setTable('orders');
        $this->setPrimaryKey('id');
        $this->setDisplayField('order_no');

        // Behaviours
        $this->addBehavior('Timestamp');

        // Associations
        $this->belongsTo('Users', [
            'foreignKey' => 'user_id',     // can be NULL for legacy/guest orders
            'joinType'   => 'LEFT',
        ]);

        $this->hasMany('OrderItems', [
            'foreignKey' => 'order_id',
            'dependent'  => true,          // delete items if order is deleted
            'cascadeCallbacks' => true,
        ]);
    }

    public function validationDefault(Validator $validator): Validator
    {
        $validator
            ->scalar('order_no')
            ->maxLength('order_no', 50)
            ->requirePresence('order_no', 'create')
            ->notEmptyString('order_no');

        $validator
            ->scalar('status')
            ->maxLength('status', 30)
            ->notEmptyString('status');

        $validator
            ->numeric('subtotal')
            ->greaterThanOrEqual('subtotal', 0)
            ->allowEmptyString('subtotal');

        $validator
            ->numeric('total')
            ->greaterThanOrEqual('total', 0)
            ->allowEmptyString('total');

        $validator->allowEmptyString('customer_name');
        $validator->email('email')->allowEmptyString('email');
        $validator->allowEmptyString('phone');

        // user_id is optional (legacy guest orders); INT UNSIGNED in DB
        $validator->nonNegativeInteger('user_id')->allowEmptyString('user_id');

        return $validator;
    }

    public function buildRules(RulesChecker $rules): RulesChecker
    {
        // If user_id is present, it must exist in users
        $rules->add($rules->existsIn(['user_id'], 'Users'), [
            'allowNullableNulls' => true,
            'errorField' => 'user_id',
            'message' => 'User not found for this order.'
        ]);

        return $rules;
    }
}



