<?php
/**
 * Routes configuration.
 *
 * In this file, you set up routes to your controllers and their actions.
 * Routes are very important mechanism that allows you to freely connect
 * different URLs to chosen controllers and their actions (functions).
 *
 * It's loaded within the context of `Application::routes()` method which
 * receives a `RouteBuilder` instance `$routes` as method argument.
 *
 * CakePHP(tm) : Rapid Development Framework (https://cakephp.org)
 * Copyright (c) Cake Software Foundation, Inc. (https://cakefoundation.org)
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @copyright     Copyright (c) Cake Software Foundation, Inc. (https://cakefoundation.org)
 * @link          https://cakephp.org CakePHP(tm) Project
 * @license       https://opensource.org/licenses/mit-license.php MIT License
 */

use Cake\Routing\Route\DashedRoute;
use Cake\Routing\RouteBuilder;

/*
 * This file is loaded in the context of the `Application` class.
 * So you can use `$this` to reference the application class instance
 * if required.
 */
return function (RouteBuilder $routes): void {
    /*
     * The default class to use for all routes
     *
     * The following route classes are supplied with CakePHP and are appropriate
     * to set as the default:
     *
     * - Route
     * - InflectedRoute
     * - DashedRoute
     *
     * If no call is made to `Router::defaultRouteClass()`, the class used is
     * `Route` (`Cake\Routing\Route\Route`)
     *
     * Note that `Route` does not do any inflections on URLs which will result in
     * inconsistently cased URLs when used with `{plugin}`, `{controller}` and
     * `{action}` markers.
     */
    $routes->setRouteClass(DashedRoute::class);

    $routes->scope('/', function (RouteBuilder $builder): void {
        /*
         * Here, we are connecting '/' (base path) to a controller called 'Pages',
         * its action called 'display', and we pass a param to select the view file
         * to use (in this case, templates/Pages/home.php)...
         */
        $builder->connect('/', ['controller' => 'Pages', 'action' => 'display', 'home']);

        $builder->connect('/about', ['controller' => 'Pages', 'action' => 'display', 'about']);

        $builder->connect('/contact/thank-you', 'Contact::thanks');

        $builder->connect('/auth/admindashboard', ['controller' => 'Auth', 'action' => 'adminDashboard', 'admin_dashboard']);

        $builder->connect('/product/detail', ['controller' => 'Products', 'action' => 'detail']);
        $builder->connect('/product', ['controller' => 'Products', 'action' => 'view']);
        $builder->connect('/contact', 'Contact::index');

        $builder->connect('/pages/*', 'Pages::display');

        $builder->connect('/products', ['controller' => 'Products', 'action' => 'index']);
        $builder->connect('/products/add', ['controller' => 'Products', 'action' => 'add']);
        $builder->connect('/products/edit/*', ['controller' => 'Products', 'action' => 'edit']);
        $builder->connect('/products/delete/*', ['controller' => 'Products', 'action' => 'delete']);

        $builder->connect('/shop', ['controller' => 'ProductBrowser', 'action' => 'shop']);
        $builder->connect('/productBrowser', ['controller' => 'ProductBrowser', 'action' => 'index']);
        $builder->connect('/products/add', ['controller' => 'ProductBrowser', 'action' => 'add']);
        $builder->connect('/products/edit/*', ['controller' => 'ProductBrowser', 'action' => 'edit']);
        $builder->connect('/products/delete/*', ['controller' => 'ProductBrowser', 'action' => 'delete']);

    // Add-to-cart (mapped to ProductsController::addToCart)
        $builder->connect('/cart/add/*', ['controller' => 'Products', 'action' => 'addToCart']);

    // Cart pages
        $builder->connect('/cart', ['controller' => 'Cart', 'action' => 'index']);
        $builder->connect('/cart/update/*', ['controller' => 'Cart', 'action' => 'update']);
        $builder->connect('/cart/clear', ['controller' => 'Cart', 'action' => 'clear']);

        $builder->connect('/checkout', ['controller' => 'Orders', 'action' => 'checkout', '_name' => 'checkout']);

        // POST-only endpoint that the big "Checkout" button submits to
        $builder->post('/orders/place', ['controller' => 'Orders', 'action' => 'place', '_name' => 'order:place']);

        // Stripe Payment
        $builder->post('/orders/webhook', ['controller' => 'Orders', 'action' => 'webhook']);

        // Optional thank-you page after placing order
        $builder->connect('/order/confirmation', ['controller' => 'Orders', 'action' => 'confirmation', '_name' => 'order:confirmation']);

        $builder->scope('/admin', function (RouteBuilder $admin) {
            $admin->connect('/orders', ['controller' => 'Orders', 'action' => 'adminIndex']);
            $admin->connect('/orders/view/*', ['controller' => 'Orders', 'action' => 'adminView']);
        });




        /*
         * Connect catchall routes for all controllers.
         *
         * The `fallbacks` method is a shortcut for
         *
         * ```
         * $builder->connect('/{controller}', ['action' => 'index']);
         * $builder->connect('/{controller}/{action}/*', []);
         * ```
         *
         * It is NOT recommended to use fallback routes after your initial prototyping phase!
         * See https://book.cakephp.org/5/en/development/routing.html#fallbacks-method for more information
         */
        $builder->fallbacks();
    });


        #api
        $routes->prefix('Api', function (RouteBuilder $routes): void {
        // (Optional) parse .json URLs if you ever want /api/...json
        // $routes->setExtensions(['json']);

        // ADDED: main chatbot endpoint
        $routes->connect(
            '/chat/recommend',
            ['controller' => 'Chat', 'action' => 'recommend', '_method' => 'POST']
        );

        // Keep fallbacks inside the prefix (uses DashedRoute by default)
        $routes->fallbacks();
    });

     
};
