<?php
declare(strict_types=1);

namespace App\Controller;

use App\Model\Table\UsersTable;
use Cake\I18n\DateTime;
use Cake\Mailer\Mailer;
use Cake\Utility\Security;
use Cake\Core\Configure;
use Cake\Http\Client;
use App\Model\Table\LandingPagesTable;
use Cake\I18n\DateTime as CakeDateTime;
/**
 * Auth Controller
 *
 * @property \Authentication\Controller\Component\AuthenticationComponent $Authentication
 */
class AuthController extends AppController
{
    /**
     * @var \App\Model\Table\UsersTable $Users
     */
    private UsersTable $Users;
    private LandingPagesTable $LandingPages;
    /**
     * Controller initialize override
     *
     * @return void
     */
    public function initialize(): void
    {
        parent::initialize();

        // By default, CakePHP will (sensibly) default to preventing users from accessing any actions on a controller.
        // These actions, however, are typically required for users who have not yet logged in.
        $this->Authentication->allowUnauthenticated(['login', 'register', 'forgetPassword', 'resetPassword','view','index','display','logout']);

        // CakePHP loads the model with the same name as the controller by default.
        // Since we don't have an Auth model, we'll need to load "Users" model when starting the controller manually.
        $this->Users = $this->fetchTable('Users');
        $this->LandingPages=$this->fetchTable('LandingPages');
    }

    private function isAdmin(): bool
    {
        $id = $this->request->getAttribute('identity');
        return (bool)($id && $id->get('role') === 'admin');
    }
    private function isStaff(): bool
    {
        $id = $this->request->getAttribute('identity');
        return (bool)($id && $id->get('role') === 'staff');
    }
    /**
     * Register method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function register()
    {
        $user = $this->Users->newEmptyEntity();
        if ($this->request->is('post')) {
            $user = $this->Users->patchEntity($user, $this->request->getData());
            if ($this->Users->save($user)) {
                $this->Flash->success('You have been registered. Please log in. ');

                return $this->redirect(['action' => 'login']);
            }
            $this->Flash->error('The user could not be registered. Please, try again.');
        }
        $this->set(compact('user'));
    }

    /**
     * Forget Password method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful email send, renders view otherwise.
     */
    public function forgetPassword()
    {
        if ($this->request->is('post')) {
            $email = $this->request->getData('email');
            $user = $this->Users->findByEmail($email)->first();
            $token = Security::hash(Security::randomBytes(32), 'sha256', true);
            $expiry = CakeDateTime::now()->modify('+1 hour')->format('Y-m-d H:i:s');
            if ($user) {
                $user->reset_token = $token;
                $user->token_expiry = $expiry;
                $this->Users->updateAll(
                    ['reset_token' => $token, 'token_expiry' => $expiry],
                    ['id' => $user->id]
                );
                $this->Users->save($user);
            }
            $resetLink = "https://brewhub.u25s2212.iedev.org/auth/reset-password?token={$token}&email={$email}";
            $mailer = new Mailer('default');
            $mailer->setFrom(['brewhub@u25s2212.iedev.org' => 'BrewHub'])
                ->setTo('brewhub@u25s2212.iedev.org')
                ->setSubject('Password Reset Request')
                ->deliver("A password reset was requested for: {$email}\n\nClick the link to reset your password:\n{$resetLink}");

            $this->Flash->success('If the email exists in our system, a reset link has been sent. Please check your email.');
            return $this->redirect(['action' => 'login']);
        }
    }


    /**
     * Change Password method
     *
     * @param string|null $id User id.
     * @return \Cake\Http\Response|null|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function resetPassword()
    {
        $token = $this->request->getQuery('token');
        $user = $this->Users->find()
            ->where([
                'reset_token' => $token,
                'token_expiry >=' => CakeDateTime::now()->format('Y-m-d H:i:s')
            ])
            ->first();

        if (!$user) {
            $this->Flash->error('Invalid or expired reset link.');
            return $this->redirect(['action' => 'forgetPassword']);
        }

        if ($this->request->is(['post', 'put'])) {
            $user = $this->Users->patchEntity($user, $this->request->getData(), ['validate' => 'resetPassword']);
            $user->reset_token = null;
            $user->token_expiry = null;
            if ($this->Users->save($user)) {
                $this->Flash->success('Password reset successfully.');
                return $this->redirect(['action' => 'login']);
            }
            $this->Flash->error('Could not reset password.');
        }

        $this->set(compact('user'));
    }


    /**
     * Login method
     *
     * @return \Cake\Http\Response|null|void Redirect to location before authentication
     */
    public function login()
    {
        $this->request->allowMethod(['get', 'post']);

        if ($this->request->is('post')) {
            $captcha = (string)$this->request->getData('g-recaptcha-response');

            if ($captcha === '') {
                $this->Flash->error('Please complete the CAPTCHA.');
                return null;
            }

            $http = new Client(['timeout' => 10]);
            $secret = (string)Configure::read('Recaptcha.secret');

            $res = $http->post('https://www.google.com/recaptcha/api/siteverify', [
                'secret' => $secret,
                'response' => $captcha,
            ]);
            $data = (array)json_decode((string)$res->getStringBody(), true);

            if (empty($data['success'])) {
                $this->Flash->error('CAPTCHA failed. Please try again.');
                return null; // render login view again
            }

            $result = $this->Authentication->getResult();

            // if user passes authentication, grant access to the system
            if ($result && $result->isValid()) {
                $user = $this->request->getAttribute('identity');
                if ($user->get('role') === 'customer') {
                    return $this->redirect(['controller' => 'Pages', 'action' => 'display', 'home']);
                }
                // set a fallback location in case user logged in without triggering 'unauthenticatedRedirect'
                $fallbackLocation = ['controller' => 'auth', 'action' => 'adminDashboard'];

                // and redirect user to the location they're trying to access
                return $this->redirect($this->Authentication->getLoginRedirect() ?? $fallbackLocation);
            }

            // display error if user submitted their credentials but authentication failed
            if ($this->request->is('post') && !$result->isValid()) {
                $this->Flash->error('Email address and/or Password is incorrect. Please try again. ');
            }
        }
    }

    /**
     * Logout method
     *
     * @return \Cake\Http\Response|null|void
     */
    public function logout()
    {
        // We only need to log out a user when they're logged in
        $result = $this->Authentication->getResult();
        if ($result && $result->isValid()) {
            $this->Authentication->logout();

            $this->Flash->success('You have been logged out successfully. ');
        }

        // Otherwise just send them to the login page
        return $this->redirect(['controller' => 'Pages', 'action' => 'display','home']);
    }

    public function inquiries()
    {
        if (!$this->isStaff() && !$this->isAdmin()) {
            throw new \Cake\Http\Exception\ForbiddenException('Staffs only.');
        }
        $submissions = $this->fetchTable('ContactSubmissions')
        ->find()
        ->orderBy(['created' => 'DESC'])
        ->all();
        $result = $this->Authentication->getResult();
        if (!$result || !$result->isValid()) {
            $this->Flash->error('Please login to view this page.');
            return $this->redirect(['controller' => 'Pages', 'action' => 'display','home']);
        }
        $user = $this->Authentication->getIdentity();
        $this->set(compact('submissions', 'user'));
    }

    public function adminDashboard()
    {
        if (!$this->isStaff() && !$this->isAdmin()) {
            throw new \Cake\Http\Exception\ForbiddenException('Staffs only.');
        }
        $user = $this->request->getAttribute('identity');
        $this->set(compact('user'));
        $this->viewBuilder()->setLayout('default');
    }

    public function editLanding()
    {
        if (!$this->isAdmin()) {
            throw new \Cake\Http\Exception\ForbiddenException('Admins only.');
        }
        $landing = $this->LandingPages->get(1);
        if ($this->request->is(['post', 'put'])) {
            $data = $this->request->getData();
            $ids = [];
            if (!empty($data['featured_products'])) {
                $ids = array_filter(array_map('intval', explode(',', $data['featured_products'])));
            }
            $existingIds = $this->fetchTable('Products')
                ->find()
                ->where(['id IN' => $ids])
                ->all()
                ->map(fn($product) => $product->id)
                ->toArray();
            $missingIds = array_diff($ids, $existingIds);
            if (!empty($missingIds)) {
                $this->Flash->error(__('The following product IDs were not found: ') . implode(', ', $missingIds));
            } else {
                $data['featured_products'] = json_encode($ids);
                $landing = $this->LandingPages->patchEntity($landing, $data);
                if ($this->LandingPages->save($landing)) {
                    $this->Flash->success(__('Landing page updated successfully.'));
                    return $this->redirect(['action' => 'editLanding']);
                }
                $this->Flash->error(__('Unable to save landing page.'));
            }
        }
        $landing->featured_products = implode(',', json_decode($landing->featured_products, true));
        $this->set(compact('landing'));
    }
}
