<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Table;
use Cake\Validation\Validator;
use Cake\Event\EventInterface;
use ArrayObject;

class OrdersTable extends Table
{
    public const STATUSES = ['waiting','approved','processing','completed','canceled'];

    public function initialize(array $config): void
    {
        parent::initialize($config);

        $this->setTable('orders');
        $this->setPrimaryKey('id');
        $this->setDisplayField('order_no');

        $this->addBehavior('Timestamp');

        $this->hasMany('OrderItems', [
            'foreignKey' => 'order_id',
            'dependent'  => true,
            'saveStrategy' => 'replace',
        ]);
    }

    public function validationDefault(Validator $v): Validator
    {
        return $v
            ->scalar('order_no')->allowEmptyString('order_no') // will auto-generate
            ->decimal('subtotal')->notEmptyString('subtotal')
            ->decimal('discount')->notEmptyString('discount')
            ->decimal('total')->notEmptyString('total')
            ->inList('status', self::STATUSES);
    }

    // Auto-generate human-friendly unique order number, e.g., BH-20250918-7F3C
    public function beforeSave(EventInterface $event, $entity, ArrayObject $options): void
    {
        if ($entity->isNew() && empty($entity->order_no)) {
            $entity->order_no = $this->generateOrderNo();
        }
    }

    protected function generateOrderNo(): string
    {
        // Adjust prefix to your brand
        $prefix = 'BH-' . date('Ymd') . '-';
        do {
            $candidate = $prefix . strtoupper(bin2hex(random_bytes(2))); // 4 hex chars
        } while ($this->exists(['order_no' => $candidate]));
        return $candidate;
    }
}


