<?php
declare(strict_types=1);

namespace App\Form;

use Cake\Form\Form;
use Cake\Form\Schema;
use Cake\Validation\Validator;
use Cake\Mailer\Mailer;
use Cake\Core\Configure;
use Cake\Http\Client;

class ContactForm extends Form
{
    protected function _buildSchema(Schema $schema): Schema
    {
        $schema
            ->addField('first_name', 'string')
            ->addField('last_name', 'string')
            ->addField('email', 'string')
            ->addField('phone', 'string')
            ->addField('message', ['type' => 'text'])
            ->addField('website', 'string') // honeypot
            ->addField('g-recaptcha-response', 'string');

        return $schema;
    }

    public function validationDefault(Validator $validator): Validator
    {
        $validator
            ->requirePresence(['first_name', 'last_name', 'email', 'message'], 'create')
            ->notEmptyString('first_name', 'First name is required')
            ->notEmptyString('last_name', 'Last name is required')
            ->email('email', false, 'Please provide a valid email')
            ->notEmptyString('message', 'Please enter a message')
            ->allowEmptyString('phone')
            ->add('phone', 'numeric', [
                'rule' => ['custom', '/^[0-9]*$/'],
                'message' => 'Phone number must contain only digits',
                'allowEmpty' => true
            ])
            ->add('phone', 'maxLength', [
                'rule' => ['maxLength', 10],
                'message' => 'Phone number must not exceed 10 digits',
                'allowEmpty' => true,
            ])
            ->add('first_name', 'alpha', [
            'rule' => ['custom', '/^[a-zA-Z]+$/'],
            'message' => 'First name must contain only letters',
            ])
            ->add('last_name', 'alpha', [
            'rule' => ['custom', '/^[a-zA-Z]+$/'],
            'message' => 'Last name must contain only letters',
            ])
            ->allowEmptyString('website');

        // ensures that user actually clicks the checkbox (is required)
        $validator->notEmptyString('g-recaptcha-response', 'Please complete the CAPTCHA.');

        // validation rule - verify CAPTCHA with Google's siteverify API
        $validator->add('g-recaptcha-response', 'recaptcha', [
            'rule' => function ($value, $context) {
                if (empty($value)) {
                    return false; // if no token, it means failure
                }
                $http = new Client(['timeout' => 10]);
                $secret = (string)Configure::read('Recaptcha.secret');

                // POST to Google's reCAPTCHA verification endpoint
                $res = $http->post(
                    'https://www.google.com/recaptcha/api/siteverify',
                    [
                        'secret' => $secret, // secret key outlined in app_local.php
                        'response' => $value, // token sent by the widget
                    ]
                );

                // decode the response and check success
                $data = (array)json_decode((string)$res->getStringBody(), true);
                return !empty($data['success']);
            },
            'message' => 'CAPTCHA failed. Please try again.' // show if verification failed and tell user to retry
        ]);

        return $validator;
    }

    protected function _execute(array $data): bool
    {
        if (!empty($data['website'])) {
            return true; // bot trap
        }
        return true;
    }
}

